<?php
/**
 *	@package		Sendinblue
 *	@subpackage		sendinblue.php
 *	@version		1.0.13
 *
 *	@author			Branislav Gligorov <branislav.gligorov@itsbg.eu>	
 *	@copyright		Copyright (C) 2018 Sendinblue, All Rights Reserved
 *	@license		GNU/GPL Version 2 or later - http://www.gnu.org/licenses/gpl-2.0.html
 */


// No direct access to this file
defined('_JEXEC') or die('Restricted access');

use \Joomla\Registry\Registry;
use \Joomla\Utilities\ArrayHelper;



 /*
  * Sendinblue api
  * 
  *  
  */
class SibJapi {
	
	
	public $baseurl = '';
	public $connectiontimeout = '';
	public $settings = null;
	public $http = null;
	public $errors = array();
	public $error_reporting = 1; // 0 = no reporting, 1 = basic reporting, 2 = all errors
	public $accountDetails = null;
	
	public function __construct($autoLoadSetting = false, $autoLoadAccountData = false)
	{
		// get params
		$app = \JFactory::getApplication();
		$params	= JComponentHelper::getParams('com_sendinblue');

		$this->baseurl = trim($params->get('baseurl', 'https://api.sendinblue.com/v3/'));
		if ($this->baseurl == '') {
			$msg = JText::_('COM_SENDINBLUE_OPTIONS_ERROR_BASEURL_NOTSET');
			$this->errors[] = $msg;
		}
		$this->connectiontimeout = trim($params->get('connectiontimeout', '30'));
		if ($this->connectiontimeout == '') {
			$msg = JText::_('COM_SENDINBLUE_OPTIONS_ERROR_BASEURL_NOTSET');
			$this->errors[] = $msg;
		}
		if ($autoLoadSetting) {
			if( ($this->settings = $this->loadSettings()) === false ) {
				$msg = JText::_('COM_SENDINBLUE_JAPI_UNABLE_TO_LOAD_SETTINGS');
				$this->errors[] = $msg;	
			}
		}
		if ($autoLoadAccountData && $this->settings !== false) {
			if( ($this->accountDetails = $this->getAccount()) === false ) {
				$msg = JText::_('COM_SENDINBLUE_JAPI_UNABLE_TO_LOAD_ACCOUNT');
				$this->errors[] = $msg;	
			}
		}
	}

	
	/**
	 * Load api key data - active or specific
	 * @return false if not faound or first active
	 */
	public function loadSettings($id = null) {
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		$query->select('*');
		$query->from($db->quoteName('#__sendinblue_setting', 'a'));
		if ($id == null) { // looking for any active
			$query->where('a.published = 1');	
		} else { // looking for specific one
			$query->where('a.id = ' . $id);
		}
		$db->setQuery($query);
		$data = $db->loadObject();
		if (empty($data)) {
			return false;
		} else {
			return $data;
		}
	}
	

	/**
	 *	http transfer (curl) wrapper for Sendinblue - setting base curl options
	 * 
	**/
	public function getHttp() {
		$curlParams = array(
			'follow_location' => true
		);
		$transportParams = new Registry($curlParams);

		try {
			$http = JHttpFactory::getHttp($transportParams, 'curl');
		} catch (RuntimeException $e) {
			$msg = JText::_('COM_SENDINBLUE_GLOBAL_ERROR') . ' - ' . JText::_('JGLOBAL_AUTH_CURL_NOT_INSTALLED');
			$this->errors[] = $msg;
			return false;
		}
		// "must have" headers
		$headers = array(
			'Content-Type' => 'application/json',
			'api-key' => $this->settings->apikey
		);
		$http->setOption('headers', $headers);
		
		return $http;
	}	


	/**
	* Retrieve Sendinblue account details data
	* @return false on error, stdClass on succcess
	**/
	public function getAccount() {
		if (!($http = $this->getHttp())) { return false; } // error msg already queued

		$headers = array(); // additional headers
		try {
			$result = $http->get($this->baseurl.'account', $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 200) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}

	/**
	*Send post install message 
	 * @return false on error, stdClass on succcess
	**/
	public function sendPostInstallMesage() {
		if (!($http = $this->getHttp())) { return false; } // error msg already queued

		$data = new stdClass();
		$data->partnerName = "joomla"; 
		$data = json_encode($data);
		
		$headers = array(); // additional headers
		try {
			$result = $http->post($this->baseurl.'account/partner', $data, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 204) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}

	

	public function getDataPage($dataid = '', $offset = 0, $limit = 1, $headers = array()) { // $headers = additional headers
		if ($dataid == '') { return false; }
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		if (strpos($dataid, '?') === false) { $operator = '?'; } else { $operator = '&'; }
		$pagination = ''; // it will be no pagination data for special case $limit == 0 
		if ($limit != 0) { $pagination = $operator .'limit=' . $limit . '&offset=' . $offset; }
		try {
			$result = $http->get($this->baseurl . $dataid . $pagination, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}

		$resultbody = json_decode($result->body);
		if ($result->code == 200) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	} 


	/*
	 * Get all data for selected source from SIB
	 * 
	 * @return 	data on success, false one error
	 */
	public function getDataAll($dataid = '', $all = false, $returndata = false, $headers = array() ) {
		if ($dataid == '') { return false; } 
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
	
		if (strpos($dataid, '/') === false) {
			if (strpos($dataid, '?') === false) {
				$datavar = $dataid;
			} else {
				$dataparts = explode('?', $dataid);
				$datavar = $dataparts[0];
			}				
		} else {
			$dataparts = explode('/', $dataid);
			$lastsegment = $dataparts[count($dataparts) - 1];
			if (strpos($lastsegment, '?') === false) {
				$datavar = $lastsegment;
			} else {
				$dataparts = explode('?', $lastsegment);
				$datavar = $dataparts[0];
			}
		} 

		$data = array(); 
	
		$step = 10; // maximum step depend on data type - have to use minimal maximum step 
		if ( $all !== false ) { $step = 0; } // get all at once -> no paginatio will be used to retrieve data 
		$offset = 0;
		$page = 0;
		$i = 0;
		$iMax = 1000; // max pages/reguests in one loop
		
		$response = $this->getDataPage($dataid, $offset, $step, $headers);
		if ($response !== false) {
			if ( $step == 0 ) { // $response->data->count does not exist for data which can not be retreived with pagination so we return data that we get from this first inquery
				if ($returndata != false) {
					$data = $response->data;
				} else {
					$data = $response->data->{$datavar};
				}
			} else {
				$total = $response->data->count;  
				//while ( ($step > $total || ($offset + $step ) < $total) && ($i < $iMax)) {
				while ( ($offset < $total) && ($i < $iMax)) {
					$offset = $step * $page;
					if ($offset < $total) {
						$response = $this->getDataPage($dataid, $offset, $step, $headers);
						if ($response !== false) {
							$data = array_merge($data, $response->data->{$datavar});
							$page++; // adds if we sucesfully get page
						}
						$i++; // always add
					}
				}
				if ($i == $iMax) {
					if (($offset + $step ) < $total) {
						return false;
					} else {
						// finished exactly at last $i  
					}
				}
			}

		} else {
			return false;
		}
		return $data;
	} 

	public function getListsAll() {} // use $response = $this->getDataAll('contacts/lists');
	public function getFoldersAll() {} // use $response = $this->getDataAll('contacts/folders'); 
	public function getContactsAll() {} // use $response = $this->getDataAll('contacts');


	/**
	* Create folder 
	* @return false on error, stdClass on succcess
	**/
	public function createFolder($foldername = null, $headers = array()) { // additional headers
		if ($foldername === null || trim($foldername) == '' ) { return false; }
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		$postdata = new stdClass();
		$postdata->name = $foldername; 
		
		$data = json_encode($postdata);
				
		try {
			$result = $http->post($this->baseurl.'contacts/folders', $data, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 201) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}
	

	/**
	* Create list 
	* @return false on error, stdClass on succcess
	**/
	public function createList($listname = null, $folderId = null, $headers = array()) { // additional headers
		if ($listname === null || trim($listname) == '' ) { return false; }
		if ($folderId === null || trim($folderId) == '' ) { return false; }
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		$postdata = new stdClass();
		$postdata->name = $listname; 
		$postdata->folderId = (int) $folderId;
		
		$data = json_encode($postdata);
				
		try {
			$result = $http->post($this->baseurl.'contacts/lists', $data, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 201) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}
	
	/**
	* Crete attribute 
	* @return false on error, stdClass on succcess
	**/
	public function createAttribute($attributeCategory = '', $attributeName = '', $data = null, $headers = array()) { // additional headers
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		// $data = new stdClass();
		// $data->value = "calculated value formula"; 
		// $data->enumeration = array(Obj1, Obj2, Obj3,...);
		// $data->type = 'text';
				
		try {
			$result = $http->post($this->baseurl.'contacts/attributes/'.$attributeCategory.'/'.$attributeName, json_encode($data), $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 201) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}


	
		
	/**
	* Crete contact 
	* @return false on error, stdClass on succcess
	**/
	public function createContact($cdata = null, $headers = array()) { // additional headers
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		// $attribs = new stdClass();
		// $attribs->FIRSTNAME = "fnmkhkhj";
		// $cdata = new stdClass();
		// $cdata->email = "test4@fsdfhhkhjsfs.com"; 
		// $cdata->emailBlacklisted = 0;
		// $cdata->attributes = $attribs;
		// $cdata->listIds = array(22,23);
		
		$data = json_encode($cdata);
				
		try {
			$result = $http->post($this->baseurl.'contacts', $data, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 201) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}

	/**
	* Update contact 
	* @return false on error, stdClass on succcess
	**/
	public function updateContact($cdata = null, $headers = array()) { // additional headers
		if (!($http = $this->getHttp())) { return false; } // error msg already queued
		
		// $attribs = new stdClass();
		// $attribs->FIRSTNAME = "fnmkhkhj";
		// $cdata = new stdClass();
		// $cdata->email = "test4@fsdfhhkhjsfs.com"; 
		// $cdata->emailBlacklisted = 0;
		// $cdata->attributes = $attribs;
		// $cdata->listIds = array(22,23);
		
		$data = json_encode($cdata);
		try {
			$result = $http->put($this->baseurl.'contacts/' . urlencode(strtolower($cdata->email)), $data, $headers, $this->connectiontimeout);
		} catch (Exception $e) { // comunincation error
			$msg = JText::_('COM_SENDINBLUE_CURL_ERROR');
			if (JDEBUG) { $msg .= ' - ' . $e; }
			$this->errors[] = $msg;
			return false;
		}
		$resultbody = json_decode($result->body);
		if ($result->code == 204) {
			$response = new stdClass;
			$response->data = $resultbody;
		} else {
			$this->errors[] = $resultbody->message . ' - ' . $resultbody->code . ' (' . $result->code .')';
			$response = false;
		}
		return $response;
	}

	

}
