<?php
/**
 *	@package		Sendinblue
 *	@subpackage		sendinblue.php
 *	@version		1.0.13
 *
 *	@author			Branislav Gligorov <branislav.gligorov@itsbg.eu>	
 *	@copyright		Copyright (C) 2018 Sendinblue, All Rights Reserved
 *	@license		GNU/GPL Version 2 or later - http://www.gnu.org/licenses/gpl-2.0.html
 */


// No direct access to this file
defined('_JEXEC') or die('Restricted access');


use \Joomla\Registry\Registry;

/**
 * Sendinblue component helper.
 */
abstract class SibJapiHelper {
	
	/**
	 * Helper fn to instantiate SibJapi with autoloading settings and account data
	 * (some kind of shortcut to see if we can work with SibJapi)
	 * @return SibJapi object or false if there are errors with throwing error to app
	 */
	public static function getSibJapi() {
		$app = \JFactory::getApplication();
		$sibjapi = new SibJapi(true, true);
		if (count($sibjapi->errors)) {
			if ($sibjapi->error_reporting > 0 ) { foreach ($sibjapi->errors as $key => $value) { $app->enqueueMessage($value, 'error'); } }
			return false;
		}
		return $sibjapi;		
	}


	/**
	 * Retrieve Sendinblue user lists
	 * @return array with id => value pairs or array with error element
	 */
	public static function getListsSelectData() {
		$lists = array();
		if ($sibjapi = self::getSibJapi()) {
			$listsResponse = $sibjapi->getDataAll('contacts/lists');
			$foldersResponse = $sibjapi->getDataAll('contacts/folders');
			if ($listsResponse && $foldersResponse) {
				foreach ($foldersResponse as $item) { $folders[$item->id] = $item->name; }
				foreach ($listsResponse as $item) { $lists[$item->id] = $folders[$item->folderId] . '/' . $item->name; }
				asort($lists, SORT_NATURAL | SORT_FLAG_CASE);
			}
		} else {
			$lists['error'] = '(Unable to get data from server)';
		}
		return $lists;		
	}

	/**
	 * Retrieve Sendinblue user templates
	 * @return array with id => value pairs or array with error element
	 */
	public static function getTemplatesSelectData() {
		$data = array();
		if ($sibjapi = self::getSibJapi()) {
			$dataResponse = $sibjapi->getDataAll('smtp/templates?templateStatus=true');
			foreach ($dataResponse as $item) { $data[$item->id] = $item->name; }
			asort($data, SORT_NATURAL | SORT_FLAG_CASE);
		} else {
			$data['error'] = '(Unable to get data from server)';
		}
		return $data;		
	}

	/**
	 * Retrieve Sendinblue user attributes
	 * @param	string 		attribute category
	 * @param	string		attribute type
	 * @return array with id => value pairs or array with error element
	 */
	public static function getAttributesSelectData($category = '', $type = '') {
		$data = array();
		if ($sibjapi = self::getSibJapi()) {
			$dataResponse =  $sibjapi->getDataAll('contacts/attributes', true);
			foreach ($dataResponse as $item) {
				if ($category == '' && $type == '') {
					$data[$item->id] = $item->name;	
				} else {
					if ($category != '' && $type != '') {
						if ($item->category == $category && $item->type == $type) { $data[$item->id] = $item->name; }
					} else {
						if ($category != '' && $type == '') {
							if ($item->category == $category) { $data[$item->id] = $item->name; }
						} else {
							if ($item->type == $type) { $data[$item->id] = $item->name; }
						}
					}
				}
			}
			asort($data, SORT_NATURAL | SORT_FLAG_CASE);
		} else {
			$data['error'] = '(Unable to get data from server)';
		}
		return $data;		
	}

	/**
	*Send post install message 
	 * @return false on error, stdClass on succcess
	**/
	public static function sendPostInstallMesage($sibjapi) {
		$db = JFactory::getDbo();
		$nullDate = $db->getNullDate();
		if ($sibjapi->settings->postinstallmsgsent != $nullDate) { return true; } // message already sent
		
		if (($sent = $sibjapi->sendPostInstallMesage()) !== false) {
			$query = $db->getQuery(true);
			$now = JFactory::getDate()->toSql();
			$query = 'UPDATE #__sendinblue_setting SET postinstallmsgsent = "' . $now . '"';
			$query .= ' WHERE id = ' . $sibjapi->settings->id;
			$db->setQuery($query);
			if ($result = $db->query()) {
				$sibjapi->settings->postinstallmsgsent = $now;
				return true;
			} 
		}
		return false;
	}


	/**
	 * Compare J mail settings with sendInBlue setings 
	 * @return false on error, stdClass on succcess
	 */
	public static function checkJmail($accountData) {
		$app = \JFactory::getApplication();
		$params	= JComponentHelper::getParams('com_sendinblue');

		$config = JFactory::getConfig();

		$mailer = $config->get('mailer');

		if ($config->get('mailer') != 'smtp') { return false; }
		if ($config->get('smtphost') != $accountData->relay->data->relay) { return false; }
		if ($config->get('smtpauth') != 1) { return false; }
		if ($config->get('smtpuser') != $accountData->relay->data->userName) { return false; }
		if ($config->get('smtpsecure') != 'tls') { return false; }
		if ($config->get('smtpport') != $accountData->relay->data->port) { return false; } // 587
		if (trim($config->get('smtppass')) == '') { return false; } 
		// only password can not be checked as it is not sent with account data
		return true;
	}

	/**
	* Set J mail settings to sendInBlue setings 
	* @return false on error, stdClass on succcess
	**/
	public static function setSMTPtoSendInBlue($sibjapi, $password) {
		// get current config
		JLoader::import( 'cms', JPATH_SITE . '/components/com_config/model' );
		JLoader::import( 'form', JPATH_SITE . '/components/com_config/model' );
		JLoader::import( 'application', JPATH_ADMINISTRATOR . '/components/com_config/model' );
		$model = JModelLegacy::getInstance( 'Application', 'ConfigModel' );
		$data = $model->getData();

		// create backup
		if ($sibjapi->settings->smtpbackup == '') {
			$backup = array();
			$backup['mailonline'] = $data['mailonline']; 
			$backup['mailer'] = $data['mailer'];
			$backup['mailfrom'] = $data['mailfrom'];
			$backup['smtpauth'] = $data['smtpauth'];
			$backup['smtpuser'] = $data['smtpuser'];
			$backup['smtppass'] = $data['smtppass'];
			$backup['smtphost'] = $data['smtphost'];
			$backup['smtpsecure'] = $data['smtpsecure'];
			$backup['smtpport'] = $data['smtpport'];
			$db = JFactory::getDbo();
			$query = $db->getQuery(true);
			$query = 'UPDATE #__sendinblue_setting SET smtpbackup = "' . $db->escape(json_encode($backup)) . '"';
			$query .= ' WHERE id = ' . $sibjapi->settings->id;
			$db->setQuery($query);
			if (!($result = $db->query())) {
				return false;
			} 
		}
		
		// set new config values
		$data['mailonline'] = 1; 
		$data['mailer'] = 'smtp';
		$data['mailfrom'] = $sibjapi->accountDetails->data->relay->data->userName;
		$data['smtpauth'] = 1;
		$data['smtpuser'] = $sibjapi->accountDetails->data->relay->data->userName;
		$data['smtppass'] = $password;
		$data['smtphost'] = $sibjapi->accountDetails->data->relay->data->relay;
		$data['smtpsecure'] = 'tls';
		$data['smtpport'] = $sibjapi->accountDetails->data->relay->data->port;

		$result = $model->save($data);	
		
		return $result;
	}
	
	public static function setJoomlaMailSettings($data) {
		// get current config
		JLoader::import( 'cms', JPATH_SITE . '/components/com_config/model' );
		JLoader::import( 'form', JPATH_SITE . '/components/com_config/model' );
		JLoader::import( 'application', JPATH_ADMINISTRATOR . '/components/com_config/model' );
		$model = JModelLegacy::getInstance( 'Application', 'ConfigModel' );
		$config = $model->getData();
		
		// set new config values
		$config['mailonline'] = $data['mailonline']; 
		$config['mailer'] = $data['mailer'];
		$config['mailfrom'] = $data['mailfrom'];
		$config['smtpauth'] = $data['smtpauth'];
		$config['smtpuser'] = $data['smtpuser'];
		$config['smtppass'] = $data['smtppass'];
		$config['smtphost'] = $data['smtphost'];
		$config['smtpsecure'] = $data['smtpsecure'];
		$config['smtpport'] = $data['smtpport'];

		$result = $model->save($config);	
		
		return $result;
		
	}

	/**
	* Update plugin state
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function setPluginTrackingID($group, $name) {
		$app	= \JFactory::getApplication();
		$db 	= \JFactory::getDbo();
		$newparams = '';
		$sibjapi = new SibJapi(true, true); 
		if ($sibjapi->accountDetails === null || $sibjapi->accountDetails === false) {
			foreach ($sibjapi->errors as $key => $value) { $app->enqueueMessage($value, 'error'); }
			// show additiona error message in view based on the object state
		} else { 
			if (isset($sibjapi->accountDetails->data->marketingAutomation->enabled) && $sibjapi->accountDetails->data->marketingAutomation->enabled == true) { // SendInblueAccount has enabled MA
				// Is plug-in instaled?
				$pluginState = SibJapiHelper::getPluginState('system', 'sendinbluema');
				if ($pluginState !== null) { // check key
					$plugin = SibJapiHelper::getPluginData('system', 'sendinbluema');
					$params = new JRegistry($plugin->params);
					$params->set('code', $sibjapi->accountDetails->data->marketingAutomation->key);
					$newparams = $params->toString();
				}
			} 
		}
		if ($newparams != '') {
			$query = 'UPDATE #__extensions SET `params` = "' . $db->escape($newparams) . '" WHERE `type` = "plugin" AND `folder` = "' . $group . '" AND `element` = "' . $name . '"';
			$db->setQuery($query);
			$result = $db->query();
		}
		return $result;	
	}


	/**
	* Update plugin state
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function setPluginState($group, $name, $enabled = 0) {
		$db = JFactory::getDbo();
		$query = 'UPDATE #__extensions SET `enabled` = ' . $enabled . ' WHERE `type` = "plugin" AND `folder` = "' . $group . '" AND `element` = "' . $name . '"';
		$db->setQuery($query);
		$result = $db->query();
		return $result;	
	}

	/**
	* Checkif plugin is intaled and enabled
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function getPluginState($group, $name) {
		$db = JFactory::getDbo();
		$query = 'SELECT * FROM #__extensions WHERE `type` = "plugin" AND `folder` = "' . $group . '" AND `element` = "' . $name . '"';
		$db->setQuery($query);
		$plugin = $db->loadObject();
		if ($plugin !== null) {
			$plugin = (bool) $plugin->enabled;
		}
		return $plugin;	
	}
	/**
	* Get plugin data
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function getPluginData($group, $name) {
		$db = JFactory::getDbo();
		$query = 'SELECT * FROM #__extensions WHERE `type` = "plugin" AND `folder` = "' . $group . '" AND `element` = "' . $name . '"';
		$db->setQuery($query);
		$plugin = $db->loadObject();
		return $plugin;	
	}

	/**
	* Get form data by id
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function getFormData($id, $published = 1) {
		$db = JFactory::getDbo();
		$query = 'SELECT * FROM #__sendinblue_subscription_form WHERE id = ' . $id . ' AND `published` = 1'; 
		$db->setQuery($query);
		$data = $db->loadObject();
		return $data;	
	}
	/**
	* Get form data by module id
	* @return null = not installed, (true or false) = enabled state
	**/
	public static function getFormDataByModuleId($moduleid, $formpublished = 1) {
		$module = JModuleHelper::getModuleById($moduleid);
		if ($module) {
			$moduleParams = new JRegistry();
		    $moduleParams->loadString($module->params);
		    $formid = $moduleParams->get('subscriptionformid', '0');
			if ($formid != '0') {
				$data = self::getFormData($formid, $formpublished);
				return $data;		
			}
		} 
		return null;
		
	}
	

	public static function getTempListId($tempFolderName = '', $tempListName = '') {
		$app	= \JFactory::getApplication();
		$params	= JComponentHelper::getParams('com_sendinblue');
		if ($tempFolderName == '') { $tempFolderName = $params->get('doubleoptintempfolder', 'FORM'); }
		if ($tempListName == '') { $tempListName = $params->get('doubleoptintemplist', 'Temp - DOUBLE OPTIN'); }
		
		$sibjapi = new SibJapi(true, true); 
		if ($sibjapi->accountDetails === null || $sibjapi->accountDetails === false) {
			foreach ($sibjapi->errors as $key => $value) { $app->enqueueMessage($value, 'error'); }
			return false;
		} else {
			$foldersResponse = $sibjapi->getDataAll('contacts/folders');
			if ($foldersResponse) { // what is response if there are zero folders?
				foreach ($foldersResponse as $item) {
					if ($item->name == $tempFolderName) { $folderId = $item->id; }
				}
				if (isset($folderId)) {
					$listsResponse = $sibjapi->getDataAll('contacts/lists');
					if ($listsResponse) {  // what is response if there are zero lists?
						foreach ($listsResponse as $item) {
							if ($item->name == $tempListName && $item->folderId == $folderId) { $listId = $item->id; }
						}
						if (isset($listId)) {
							return $listId;
						} else { // create list
							$createListResponse = $sibjapi->createList($tempListName, $folderId);
							if ($createListResponse == false) {
								if (JDEBUG) {
									$msg = 'Unable to create list';
									$sibjapi->errors[] = $msg;
								}
								return false;
							} else {
								$listId = $createListResponse->data->id;
							}
						}
					} else {
						if (JDEBUG) {
							$msg = 'Unable to get lists';
							$sibjapi->errors[] = $msg;
						}
						return false;
					}
				} else { // create folder and create list
					$createFolderResponse = $sibjapi->createFolder($tempFolderName); 
					if ($createFolderResponse == false) {
						if (JDEBUG) {
							$msg = 'Unable to create folder';
							$sibjapi->errors[] = $msg;
						}
						return false;
					} else {
						$folderId = $createFolderResponse->data->id;
						// create list
						$createListResponse = $sibjapi->createList($tempListName, $folderId);
						if ($createListResponse == false) {
							if (JDEBUG) {
								$msg = 'Unable to create list';
								$sibjapi->errors[] = $msg;
							}
							return false;
						} else {
							$listId = $createListResponse->data->id;
						}
					}
				}
			} else {
				$msg = 'Unable to get folders';
				$app->enqueueMessage($msg, 'error'); 
				return false;
			}	
		}
		return $listId;		
	}
	
	/**
	 * check if Sendinblue account has DoubleOptin Attribute if not it will be created
	 * @reurn	false on error, if attribute exist attribute->id, if attribute created true
	 */
	public static function checkDoubleOptinAttribute($sibjapi = false, $attributeName = '') {
		if ($attributeName == '') { $attributeName = 'DOUBLE_OPT-IN'; }
		if ($sibjapi == false) { $sibjapi = self::getSibJapi(); } // if nor supplied try to get new one
		if ($sibjapi == false) { return false; }
		$found = false;
		$dataResponse =  $sibjapi->getDataAll('contacts/attributes', true);
		$attr = strtoupper($attributeName);
		foreach ($dataResponse as $item) {
			if ($item->name == $attr) {
				$found = true;
				break;
			}
		}
		if ($found === false) {
			$attributeCategory = 'category';
			$data = new stdClass;
			$data->type = 'category';
			$opt1 = new stdClass; $opt2 = new stdClass;
			$opt1->value = 1; $opt1->label = 'Yes';
			$opt2->value = 2; $opt2->label = 'No';
			$data->enumeration = array($opt1, $opt2); // [{"value":1,"label":"Yes"},{"value":2,"label":"No"}]
			if ($sibjapi->createAttribute($attributeCategory, $attributeName, $data) !== false ) {
				$found = true;
			} else {
				$msg = 'Unable to create attribute';
				$sibjapi->errors[] = $msg;
			}
		}
		return $found;
	}

	public static function createSubscriptionData($email, $formId, $confirmationtype) {
		self::cleanSubscriptionData($email, $formId);
		$db = JFactory::getDbo();
		JTable::addIncludePath(JPATH_COMPONENT_ADMINISTRATOR . '/tables');
		if ( !($row = JTable::getInstance('Subscription_data', 'SendinblueTable', array()))) {
			return false;
		}
		$row->email= $email;
		$row->subscriptionformid = $formId;
		$row->confirmationtype = $confirmationtype;
		if ($confirmationtype == '3') { // it is token value until confirmed when it is set to 1
			$row->token = bin2hex(openssl_random_pseudo_bytes(128)); 
		} else {
			$row->token = 1;
		}
		$row->published = true;
		$row->created_by = JFactory::getUser()->id;
		$row->created = date($db->getDateFormat());
		if (!$row->store()) {
			// log
			return false;
		}
		return $row->token;
	}
	public static function cleanSubscriptionData($email, $formId) {
		$db = JFactory::getDbo();
		$query = 'DELETE FROM #__sendinblue_subscription_data WHERE subscriptionformid = ' . $formId . ' AND `email` = "' . $email . '" AND token <> "1" AND confirmationtype = 3';
		$db->setQuery($query);
		$db->query();
	}


	public static function test() {
		$items = SibJapiHelper::getListsSelectData();
		if (count($items)) {
			//$options[] = JHtml::_('select.option', '', 'Select an option');
			foreach($items as $key => $value) { $options[] = JHtml::_('select.option', $key, $value); }
		}		
		return $options;		
	}
	
	/**
	 * Function used to refresh attributes on every start of editting some subscrition form (we should have fresh data from SendinBlue)
	 * when editting is done subscription modules use attributes local table to show forms
	 */
	public static function importAttribs() {
		$db = JFactory::getDbo();
		// cleanup
		$query = "TRUNCATE TABLE #__sendinblue_attribute";
		$db->setQuery($query);
		if (!($db->query())) {	// TODO log 
			return false;  
		}
		// import
		$import = true;
		if ($sibjapi = self::getSibJapi()) {
			$attrResponse = $sibjapi->getDataAll('contacts/attributes', true);
			// asort($attrs, SORT_NATURAL | SORT_FLAG_CASE);
			if ($attrResponse !== false) {
				JTable::addIncludePath(JPATH_COMPONENT_ADMINISTRATOR . '/tables');
				$i = 1;	
				foreach ($attrResponse as $item) {
					$row = JTable::getInstance('Attribute', 'SendinblueTable', array());
					$row->attribname = $item->name;
					if (isset($item->category)) { $row->attribcategoryid = $item->category; }
					if (isset($item->type)) { $row->attribtypeid = $item->type; }
					if (isset($item->calculatedValue)) { $row->attribcalculatedvalue = $item->calculatedValue; }
					if (isset($item->enumeration)) { $row->attribenumerationsubform = json_encode($item->enumeration); }
					$row->published = true;
					$row->created_by = JFactory::getUser()->id;
					$row->created = date($db->getDateFormat());
					$row->ordering = $i;
					if (!$row->store()) {
						// log
						$import = false;
					}
					$i++;
				}
			} else {
				$import = false;
			}
		} else {
			$import = false;
		}
		return $import;			
	}

	
	/**
	 * Not in  use 
	 *
	 * Function used to refresh attributes on every start of editting some subscrition form (we should have fresh data from SendinBlue)
	 * when editting is done subscription modules use attributes local table to show forms
	 */
	public static function importAttribsWithBackup() {
		$db = JFactory::getDbo();
		// first backup old attribs by setting them to unpublished
		$query = "UPDATE #__sendinblue_attribute SET `published` = 0";
		$db->setQuery($query);
		if (!($db->query())) {
			// log
			return false;
		}
		// try to import
		$import = true;
		if ($sibjapi = self::getSibJapi()) {
			$attrResponse = $sibjapi->getDataAll('contacts/attributes', true);
			// asort($attrs, SORT_NATURAL | SORT_FLAG_CASE);
			if ($attrResponse !== false) {
				JTable::addIncludePath(JPATH_COMPONENT_ADMINISTRATOR . '/tables');
				$i = 1;	
				foreach ($attrResponse as $item) {
					$row = JTable::getInstance('Attribute', 'SendinblueTable', array());
					$row->attribname = $item->name;
					$row->attribcategoryid = $item->category;
					$row->attribtypeid = $item->type;
					$row->attribcalculatedvalue = $item->calculatedValue;
					$row->attribenumerationsubform = json_encode($item->enumeration);
					$row->published = true;
					$row->created_by = JFactory::getUser()->id;
					$row->created = date($db->getDateFormat());
					$row->ordering = $i;
					if (!$row->store()) {
						// log
						$import = false;
					}
					$i++;
				}
			}
		}
		if ($import == true) { // delete old atributes 
			$query = "DELETE #__sendinblue_attribute WHERE `published` = 0";
			$db->setQuery($query);
			if (!($db->query())) {
				// log
				//return false;  no need to mark import completely false - this undeleted will be deleted with next successful import 
			}
			return true;			
		} else { // try to restore old ones
			
		}
		return false;		
	}
	
	
	
	/* next two function is just copied from ArrayHelper
	
	/**
	 * Utility function to map an object to an array
	 *
	 * @param   object   $source   The source object
	 * @param   boolean  $recurse  True to recurse through multi-level objects
	 * @param   string   $regex    An optional regular expression to match on field names
	 *
	 * @return  array
	 *
	 * @since   1.0
	 */
	public static function fromObject($source, $recurse = true, $regex = null)
	{
		if (\is_object($source) || \is_array($source))
		{
			return self::arrayFromObject($source, $recurse, $regex);
		}

		return array();
	}

	/**
	 * Utility function to map an object or array to an array
	 *
	 * @param   mixed    $item     The source object or array
	 * @param   boolean  $recurse  True to recurse through multi-level objects
	 * @param   string   $regex    An optional regular expression to match on field names
	 *
	 * @return  array
	 *
	 * @since   1.0
	 */
	private static function arrayFromObject($item, $recurse, $regex)
	{
		if (\is_object($item))
		{
			$result = array();

			foreach (get_object_vars($item) as $k => $v)
			{
				if (!$regex || preg_match($regex, $k))
				{
					if ($recurse)
					{
						$result[$k] = self::arrayFromObject($v, $recurse, $regex);
					}
					else
					{
						$result[$k] = $v;
					}
				}
			}

			return $result;
		}

		if (\is_array($item))
		{
			$result = array();

			foreach ($item as $k => $v)
			{
				$result[$k] = self::arrayFromObject($v, $recurse, $regex);
			}

			return $result;
		}

		return $item;
	}
	
	
	
	
}
	