<?php
/**
 *	@package		Sendinblue
 *	@subpackage		attributes.php
 *	@version		1.0.9
 *
 *	@author			Branislav Gligorov <branislav.gligorov@itsbg.eu>	
 *	@copyright		Copyright (C) 2018 Sendinblue, All Rights Reserved
 *	@license		GNU/GPL Version 2 or later - http://www.gnu.org/licenses/gpl-2.0.html
 */


// No direct access to this file
defined('_JEXEC') or die('Restricted access');

/**
 * Attributes Model
 */
class SendinblueModelAttributes extends JModelList
{
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
        {
			$config['filter_fields'] = array(
				'a.id','id',
				'a.published','published',
				'a.ordering','ordering',
				'a.created_by','created_by',
				'a.modified_by','modified_by',
				'a.attribname','attribname',
				'a.attribcategoryid','attribcategoryid',
				'a.attribtypeid','attribtypeid'
			);
		}

		parent::__construct($config);
	}
	
	/**
	 * Method to auto-populate the model state.
	 *
	 * @return  void
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app = JFactory::getApplication();

		// Adjust the context to support modal layouts.
		if ($layout = $app->input->get('layout'))
		{
			$this->context .= '.' . $layout;
		}
		$attribname = $this->getUserStateFromRequest($this->context . '.filter.attribname', 'filter_attribname');
		$this->setState('filter.attribname', $attribname);

		$attribcategoryid = $this->getUserStateFromRequest($this->context . '.filter.attribcategoryid', 'filter_attribcategoryid');
		$this->setState('filter.attribcategoryid', $attribcategoryid);

		$attribtypeid = $this->getUserStateFromRequest($this->context . '.filter.attribtypeid', 'filter_attribtypeid');
		$this->setState('filter.attribtypeid', $attribtypeid);
        
		$sorting = $this->getUserStateFromRequest($this->context . '.filter.sorting', 'filter_sorting', 0, 'int');
		$this->setState('filter.sorting', $sorting);
        
		$access = $this->getUserStateFromRequest($this->context . '.filter.access', 'filter_access', 0, 'int');
		$this->setState('filter.access', $access);
        
		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		$published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
		$this->setState('filter.published', $published);
        
		$created_by = $this->getUserStateFromRequest($this->context . '.filter.created_by', 'filter_created_by', '');
		$this->setState('filter.created_by', $created_by);

		$created = $this->getUserStateFromRequest($this->context . '.filter.created', 'filter_created');
		$this->setState('filter.created', $created);

		// List state information.
		parent::populateState($ordering, $direction);
	}
	
	/**
	 * Method to get an array of data items.
	 *
	 * @return  mixed  An array of data items on success, false on failure.
	 */
	public function getItems()
	{
		// load parent items
		$items = parent::getItems();

		// set selection value to a translatable value
		if (SendinblueHelper::checkArray($items))
		{
			foreach ($items as $nr => &$item)
			{
				// convert attribcategoryid
				$item->attribcategoryid = $this->selectionTranslation($item->attribcategoryid, 'attribcategoryid');
				// convert attribtypeid
				$item->attribtypeid = $this->selectionTranslation($item->attribtypeid, 'attribtypeid');
			}
		}

        
		// return items
		return $items;
	}

	/**
	 * Method to convert selection values to translatable string.
	 *
	 * @return translatable string
	 */
	public function selectionTranslation($value,$name)
	{
		// Array of attribcategoryid language strings
		if ($name === 'attribcategoryid')
		{
			$attribcategoryidArray = array(
				'global' => 'COM_SENDINBLUE_ATTRIBUTE_GLOBAL',
				'normal' => 'COM_SENDINBLUE_ATTRIBUTE_NORMAL',
				'category' => 'COM_SENDINBLUE_ATTRIBUTE_CATEGORY'
			);
			// Now check if value is found in this array
			if (isset($attribcategoryidArray[$value]) && SendinblueHelper::checkString($attribcategoryidArray[$value]))
			{
				return $attribcategoryidArray[$value];
			}
		}
		// Array of attribtypeid language strings
		if ($name === 'attribtypeid')
		{
			$attribtypeidArray = array(
				'text' => 'COM_SENDINBLUE_ATTRIBUTE_TEXT',
				'float' => 'COM_SENDINBLUE_ATTRIBUTE_FLOAT',
				'date' => 'COM_SENDINBLUE_ATTRIBUTE_DATE',
				'boolean' => 'COM_SENDINBLUE_ATTRIBUTE_BOOLEAN'
			);
			// Now check if value is found in this array
			if (isset($attribtypeidArray[$value]) && SendinblueHelper::checkString($attribtypeidArray[$value]))
			{
				return $attribtypeidArray[$value];
			}
		}
		return $value;
	}
	
	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return	string	An SQL query
	 */
	protected function getListQuery()
	{
		// Get the user object.
		$user = JFactory::getUser();
		// Create a new query object.
		$db = JFactory::getDBO();
		$query = $db->getQuery(true);

		// Select some fields
		$query->select('a.*');

		// From the sendinblue_item table
		$query->from($db->quoteName('#__sendinblue_attribute', 'a'));

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published))
		{
			$query->where('a.published = ' . (int) $published);
		}
		elseif ($published === '')
		{
			$query->where('(a.published = 0 OR a.published = 1)');
		}

		// Join over the asset groups.
		$query->select('ag.title AS access_level');
		$query->join('LEFT', '#__viewlevels AS ag ON ag.id = a.access');
		// Filter by access level.
		if ($access = $this->getState('filter.access'))
		{
			$query->where('a.access = ' . (int) $access);
		}
		// Implement View Level Access
		if (!$user->authorise('core.options', 'com_sendinblue'))
		{
			$groups = implode(',', $user->getAuthorisedViewLevels());
			$query->where('a.access IN (' . $groups . ')');
		}
		// Filter by search.
		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			else
			{
				$search = $db->quote('%' . $db->escape($search) . '%');
				$query->where('(a.attribname LIKE '.$search.')');
			}
		}

		// Filter by Attribcategoryid.
		if ($attribcategoryid = $this->getState('filter.attribcategoryid'))
		{
			$query->where('a.attribcategoryid = ' . $db->quote($db->escape($attribcategoryid)));
		}
		// Filter by Attribtypeid.
		if ($attribtypeid = $this->getState('filter.attribtypeid'))
		{
			$query->where('a.attribtypeid = ' . $db->quote($db->escape($attribtypeid)));
		}

		// Add the list ordering clause.
		$orderCol = $this->state->get('list.ordering', 'a.id');
		$orderDirn = $this->state->get('list.direction', 'asc');	
		if ($orderCol != '')
		{
			$query->order($db->escape($orderCol . ' ' . $orderDirn));
		}

		return $query;
	}
	
	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @return  string  A store id.
	 *
	 */
	protected function getStoreId($id = '')
	{
		// Compile the store id.
		$id .= ':' . $this->getState('filter.id');
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.ordering');
		$id .= ':' . $this->getState('filter.created_by');
		$id .= ':' . $this->getState('filter.modified_by');
		$id .= ':' . $this->getState('filter.attribname');
		$id .= ':' . $this->getState('filter.attribcategoryid');
		$id .= ':' . $this->getState('filter.attribtypeid');

		return parent::getStoreId($id);
	}
}
