<?php
/**
 * @package    sponsorswall
 * @subpackage models
 * @author     Alexander Grözinger {@link http://www.agmedia.de}
 * @author     Created on Mar-2018
 * @license    GNU/GPL
 * @copyright  AGMedia.de
 */
/*
   _   ___ __  __        _ _           _
  /_\ / __|  \/  |___ __| (_)__ _   __| |___
 / _ \ (_ | |\/| / -_) _` | / _` |_/ _` / -_)
/_/ \_\___|_|  |_\___\__,_|_\__,_(_)__,_\___|
*/

// No direct access to this file
defined('_JEXEC') or die(';-)');

/**
 * Methods supporting a list of banner records.
 *
 * @since  1.6
 */
class SponsorswallModelSponsors extends JModelList
{
    /**
     * Constructor.
     *
     * @param   array $config An optional associative array of configuration settings.
     *
     * @see     JControllerLegacy
     * @since   1.6
     */
    public function __construct($config = array())
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = array(
                'id', 'a.id',
                'cid', 'a.cid',
                'name', 'a.name',
                'state', 'a.state',
                'ordering', 'a.ordering',
                'catid', 'a.catid', 'category_name',
                'checked_out', 'a.checked_out',
                'checked_out_time', 'a.checked_out_time',
                'created', 'a.created',
                'clicks', 'a.clicks',
                'publish_up', 'a.publish_up',
                'publish_down', 'a.publish_down',
                'published',
            );
        }

        parent::__construct($config);
    }

    /**
     * Returns a reference to the a Table object, always creating it.
     *
     * @param   string $type The table type to instantiate
     * @param   string $prefix A prefix for the table class name. Optional.
     * @param   array $config Configuration array for model. Optional.
     *
     * @return  JTable  A JTable object
     *
     * @since   1.6
     */
    public function getTable($type = 'Sponsor', $prefix = 'SponsorswallTable', $config = array())
    {
        return JTable::getInstance($type, $prefix, $config);
    }

    /**
     * Build an SQL query to load the list data.
     *
     * @return  JDatabaseQuery
     *
     * @since   1.6
     */
    protected function getListQuery()
    {
        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Default query variables
        $db    = $this->getDbo();
        $query = $db->getQuery(true);

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Select the required fields from the table.
        $query->select(
            $this->getState(
                'list.select',
                'a.id AS id,'
                . 'a.name AS name,'
                . 'a.link AS link,'
                . 'a.image AS image,'
                . 'a.clicks AS clicks,'
                . 'a.checked_out AS checked_out,'
                . 'a.checked_out_time AS checked_out_time,'
                . 'a.cid AS cid,'
                . 'a.state AS state,'
                . 'a.ordering AS ordering,'
                . 'a.publish_up,'
                . 'a.publish_down'
            )
        );
        $query->from($db->quoteName('#__sponsorswall_sponsors', 'a'));

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Join over the users for the checked out user.
        $query->select($db->quoteName('uc.name', 'editor'))
            ->join('LEFT', $db->quoteName('#__users', 'uc') . ' ON uc.id = a.checked_out');

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Join over the categories.
        $query->select($db->quoteName('c.title', 'category_name'))
            ->join('LEFT', $db->quoteName('#__categories', 'c') . ' ON c.id = a.cid');

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Filter by published state
        $published = $this->getState('filter.published');

        if (is_numeric($published)) {
            $query->where($db->quoteName('a.state') . ' = ' . (int)$published);
        } elseif ($published === '') {
            $query->where($db->quoteName('a.state') . ' IN (0, 1)');
        }

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Filter by category.
        $categoryId = $this->getState('filter.cid');
        if (is_numeric($categoryId)) {
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            // SubQuery bauen um auch Kinder der Kategorien beim Filtern zu berücksichtigen
            $subquery = $db->getQuery(true);
            $subquery
                ->select('n.title, COUNT(*) AS level, n.id as id')
                ->from('#__categories AS n, #__categories AS p')
                ->where('n.lft BETWEEN p.lft AND p.rgt')
                ->where($db->quoteName('n.extension') . ' = ' . $db->quote('com_sponsorswall'))
                ->where($db->quoteName('p.id') . ' = ' . $db->quote($categoryId));

            $subquery->group('n.lft');
            $subquery->order('n.lft');
            $db->setQuery($subquery);
            $db->execute();
            $count  = $db->getNumRows();
            $result = $db->loadAssocList();

            ///////////////////////////////////////////////////////////////////////////////////////////////////
            // SubQuery Ergebnis auswerten und die Filterkriterien auf die Hauptquery anwenden
            if ($count > 0) {
                $categories = array();
                foreach ($result as $cat) {
                    $categories[] = $db->escape($cat['id']);
                }
                $categories_string = implode(', ', $categories);
                $query->where($db->quoteName('a.cid') . ' IN (' . $categories_string . ')');
            } else {
                $query->where($db->quoteName('a.cid') . ' = ' . (int)$categoryId);
            }
        }

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Filter by search in title
        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $query->where($db->quoteName('a.id') . ' = ' . (int)substr($search, 3));
            } else {
                $search = $db->quote('%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%'));
                $query->where('(a.name LIKE ' . $search . ')');
            }
        }

        ////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Add the list ordering clause.
        $orderCol  = $this->state->get('list.ordering', 'a.name');
        $orderDirn = $this->state->get('list.direction', 'ASC');

        if ($orderCol == 'a.ordering' || $orderCol == 'category_title') {
            $orderCol = 'c.title ' . $orderDirn . ', a.ordering';
        }

        $query->order($db->escape($orderCol . ' ' . $orderDirn));

        return $query;
    }

    /**
     * Method to get a store id based on model configuration state.
     *
     * This is necessary because the model is used by the component and
     * different modules that might need different sets of data or different
     * ordering requirements.
     *
     * @param   string $id A prefix for the store id.
     *
     * @return  string  A store id.
     *
     * @since   1.6
     */
    protected function getStoreId($id = '')
    {
        // Compile the store id.
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.published');
        $id .= ':' . $this->getState('filter.cid');

        return parent::getStoreId($id);
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @param   string $ordering An optional ordering field.
     * @param   string $direction An optional direction (asc|desc).
     *
     * @return  void
     *
     * @since   1.6
     */
    protected function populateState($ordering = 'a.ordering', $direction = 'asc')
    {
        // Load the filter state.
        $this->setState('filter.search', $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search', '', 'string'));
        $this->setState('filter.published', $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '', 'string'));
        $this->setState('filter.cid', $this->getUserStateFromRequest($this->context . '.filter.cid', 'filter_cid', '', 'cmd'));

        // Load the parameters.
        $this->setState('params', JComponentHelper::getParams('com_sponsorswall'));

        // List state information.
        parent::populateState($ordering, $direction);
    }
}
