<?php
/**
 * @package    AGMedia Library
 * @author     Alexander Grözinger {@link http://www.agmedia.de}
 * @author     Created on Apr-2018
 * @license    GNU/GPL
 * @copyright  AGMedia.de
 */
/*
   _   ___ __  __        _ _           _
  /_\ / __|  \/  |___ __| (_)__ _   __| |___
 / _ \ (_ | |\/| / -_) _` | / _` |_/ _` / -_)
/_/ \_\___|_|  |_\___\__,_|_\__,_(_)__,_\___|
*/

// No direct access to this file
defined('_JEXEC') or die(';-)');

use Joomla\Archive\Archive;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Version;

jimport('joomla.filesystem.file');
jimport('joomla.filesystem.folder');

class AgmLibDownloader
{
	protected $pro_folder_path              = '';	// Pfad in welchem die Dateien für die Pro-Erweiterungen liegen
	protected $pro_file_version				= '';	// Datei in der die aktuelle Version der Pro-Erweiterung steht
	
	protected $pro_version					= 0;	// Aktuelle Version der Pro-Erweiterung
	
	protected $update_stream_url 		    = '';	// URL des Update-Streams
	protected $zip_src_url                  = '';	// URL von welcher die Pro-Erweiterung geladen wird
	
	public function __construct($component = '')
	{
		if(!empty($component)) {
			//TODO CHECK PLAUSIBILITAET
			$this->pro_folder_path  = JPATH_ADMINISTRATOR . '/components/'.$component.'/pro';
			
			switch($component) {
				case 'com_volleyimport':
					if(defined('AGMEDIA_VOLLEYIMPORT_PRO_UPDATESTREAM')) {
						$this->setUpdateStream(AGMEDIA_VOLLEYIMPORT_PRO_UPDATESTREAM);
					}
					break;
			}
			
		} else {
			$this->pro_folder_path  = JPATH_COMPONENT_ADMINISTRATOR . '/pro';
		}
		
		
		// Set-Up directories
		$this->pro_file_version = $this->pro_folder_path . '/version.txt';
		
		$this->pro_version 		= $this->getCurrentVersion();
		
		return true;
	}
	
	public function setUpdateStream($stream) {
		if(!empty($stream)) {
			$this->update_stream_url = $stream;
		}
	}	
	
	public function getVersion() {
		return $this->pro_version;
	}
	
	/**
	 * Funktion prüft ob für die Pro-Erweiterung ein Update verfügbar ist
	 */ 
	public function checkUpdateAvailable() {
		try {
			$this->checkUpdateStreamSet();
			//Get updates from update_stream
	        $updates = $this->getUpdateArray($this->update_stream_url);
			if(empty($updates)) return false;
			
			//Get current version
			$this->getCurrentVersion();
			
			//Get Update 
			$update = $updates[0];
	        
	        // Vergleiche aktuelle Version, mit Version die auf dem Server liegt
	        $compare_result = AgmLibBase::versionCompare($update['version'], '>', $this->pro_version);
	        switch($compare_result) {
        		case 1:
        		case true:
        			$this->zip_src_url = $update['downloads'][0]['url'];
        			return true;
        			break;
	        	case -1:
	        	case 0:
	        	default:
	        		return false;
	        		break;
	        }
	        
		} catch (Exception $e) {
		    $msg 	= $e->getMessage(); // Returns "Normally you would have other code...
		    $code 	= $e->getCode(); 	// Returns '500';
		    JFactory::getApplication()->enqueueMessage($msg, 'error'); // commonly to still display that error
		}		
	}
	
	/**
	 * Funktion führt Update aus
	 */ 
	public function executeUpdate() {
		try {
			$this->checkUpdateStreamSet();
			$file   = $this->downloadPackage($this->zip_src_url);
			$unpack = $this->unpack($file, true);
			$copycontent = $this->copyContent($unpack['extractdir']);
			
		} catch (Exception $e) {
		    $msg 	= $e->getMessage(); // Returns "Normally you would have other code...
		    $code 	= $e->getCode(); 	// Returns '500';
		    JFactory::getApplication()->enqueueMessage($msg, 'error'); // commonly to still display that error
		    return false;
		}	
		
		return true;
	}
	
	private function checkUpdateStreamSet() {
		try {
			/////////////////////////////////////////////////////////////////////////////////////////////////////
			// Prüfe ob update_stream übergeben wurde
		    if(empty($this->update_stream_url)) {
		    	return false;
		    	// throw new Exception('AgmLibLoader: No update stream provided', 500);
		    }
		    
		    return true;
		} catch (Exception $e) {
		    $msg 	= $e->getMessage(); // Returns "Normally you would have other code...
		    $code 	= $e->getCode(); 	// Returns '500';
		    JFactory::getApplication()->enqueueMessage($msg, 'error'); // commonly to still display that error
		    
		    return false;
		}		
	}	
	
	
	private function getCurrentVersion() {
	    if(!JFolder::exists($this->pro_folder_path)) {
	    	// throw new Exception("Pro folder not found");
	    	return '0.0.0';
	    }	
	    
	    if(!JFile::exists($this->pro_file_version)) {
	    	return '0.0.0';
	    	// throw new Exception("Pro version file not found");
	    }		    
	    
	    //Read the file
	    $file_content = file_get_contents($this->pro_file_version);
	    
	    $regex = '/[0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,2}/i';
	    $match = array();
	    preg_match($regex, $file_content, $match);
	    if(empty($match)) {
	    	throw new Exception("No valid version of pro extension");
	    }
	    
	    return $match[0];
	}
	
	
	private function getUpdateArray($update_stream_url) {
        // Initialise
		$ret = array();
		
		//Check if updatestream is http or https
        $ssl = stripos($update_stream_url,'https://') === 0 ? true : false;
        
        //Initiate downloader
		$downloader = new FOFDownload();
		switch($downloader->getAdapterName()) {
			case "curl":
				$downloader->setAdapterOptions([
				    CURLOPT_FOLLOWLOCATION => 0,
				    //CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; MSIE 5.01; Windows NT 5.0)',
				]);
				
				if($ssl) {
			        //support https
			        $downloader->setAdapterOptions([
			        	CURLOPT_SSL_VERIFYHOST => false,
			        	CURLOPT_SSL_VERIFYPEER => false,
			        ]);
				}
				break;
			default:
				return $ret;
				break;
		}
		
		//Try to get content of update_stream_url and check if its xml
		try
		{
		    $xmlSource  = $downloader->getFromURL($update_stream_url);
		    $xml        = new SimpleXMLElement($xmlSource, LIBXML_NONET);
		}
		catch(Exception $e)
		{
			return $ret;
		}
		
		// Sanity check
		if (($xml->getName() != 'updates'))
		{
			unset($xml);
			return $ret;
		}
		
		// Let's populate the list of updates
		/** @var SimpleXMLElement $update */
		foreach ($xml->children() as $update)
		{
			// Sanity check
			if ($update->getName() != 'update')
			{
				continue;
			}
			$entry = array(
				'infourl'			=> array('title' => '', 'url' => ''),
				'downloads'			=> array(),
				'tags'				=> array(),
				'targetplatform'	=> array(),
			);
			$properties = get_object_vars($update);
			foreach ($properties as $nodeName => $nodeContent)
			{
				switch ($nodeName)
				{
					default:
						$entry[$nodeName] = $nodeContent;
						break;
					case 'infourl':
					case 'downloads':
					case 'tags':
					case 'targetplatform':
						break;
				}
			}
			$infourlNode = $update->xpath('infourl');
			$entry['infourl']['title'] = (string)$infourlNode[0]['title'];
			$entry['infourl']['url'] = (string)$infourlNode[0];
			$downloadNodes = $update->xpath('downloads/downloadurl');
			foreach ($downloadNodes as $downloadNode)
			{
				$entry['downloads'][] = array(
					'type'		=> (string)$downloadNode['type'],
					'format'	=> (string)$downloadNode['format'],
					'url'		=> (string)$downloadNode,
				);
			}
			$tagNodes = $update->xpath('tags/tag');
			foreach ($tagNodes as $tagNode)
			{
				$entry['tags'][] = (string)$tagNode;
			}
			/** @var SimpleXMLElement[] $targetPlatformNode */
			$targetPlatformNode = $update->xpath('targetplatform');
			$entry['targetplatform']['name'] = (string)$targetPlatformNode[0]['name'];
			$entry['targetplatform']['version'] = (string)$targetPlatformNode[0]['version'];
			$client = $targetPlatformNode[0]->xpath('client');
			$entry['targetplatform']['client'] = (is_array($client) && count($client)) ? (string)$client[0] : '';
			$folder = $targetPlatformNode[0]->xpath('folder');
			$entry['targetplatform']['folder'] = is_array($folder) && count($folder) ? (string)$folder[0] : '';
			$ret[] = $entry;
		}
		unset($xml);
		return $ret;
	}
	
	
	
	/**
	 * Downloads a package
	 * 
	 * FROM JOOMLA InstallerHelper ADAPTED TO MY REQUIREMENTS
	 *
	 * @param   string  $url     URL of file to download
	 * @param   mixed   $target  Download target filename or false to get the filename from the URL
	 *
	 * @return  string|boolean  Path to downloaded package or boolean false on failure
	 *
	 * @since   3.1
	 */
	private function downloadPackage($url, $target = false)
	{
		// Capture PHP errors
		$track_errors = ini_get('track_errors');
		ini_set('track_errors', true);

		// Set user agent
		$version = new Version;
		ini_set('user_agent', $version->getUserAgent('Installer'));

		// Load installer plugins, and allow URL and headers modification
		$headers = array();

		try
		{
			$response = \JHttpFactory::getHttp()->get($url, $headers);
		}
		catch (\RuntimeException $exception)
		{
			\JLog::add(\JText::sprintf('JLIB_INSTALLER_ERROR_DOWNLOAD_SERVER_CONNECT', $exception->getMessage()), \JLog::WARNING, 'jerror');

			return false;
		}

		if (302 == $response->code && isset($response->headers['Location']))
		{
			return $this->downloadPackage($response->headers['Location']);
		}
		elseif (200 != $response->code)
		{
			\JLog::add(\JText::sprintf('JLIB_INSTALLER_ERROR_DOWNLOAD_SERVER_CONNECT', $response->code), \JLog::WARNING, 'jerror');

			return false;
		}

		// Parse the Content-Disposition header to get the file name
		if (isset($response->headers['Content-Disposition'])
			&& preg_match("/\s*filename\s?=\s?(.*)/", $response->headers['Content-Disposition'], $parts))
		{
			$flds = explode(';', $parts[1]);
			$target = trim($flds[0], '"');
		}

		$tmpPath = \JFactory::getConfig()->get('tmp_path');

		// Set the target path if not given
		if (!$target)
		{
			$target = $tmpPath . '/' . self::getFilenameFromUrl($url);
		}
		else
		{
			$target = $tmpPath . '/' . basename($target);
		}

		// Write buffer to file
		\JFile::write($target, $response->body);

		// Restore error tracking to what it was before
		ini_set('track_errors', $track_errors);

		// Bump the max execution time because not using built in php zip libs are slow
		@set_time_limit(ini_get('max_execution_time'));

		// Return the name of the downloaded package
		return basename($target);
	}	
	
	
	/**
	 * FROM JOOMLA InstallerHelper ADAPTED TO MY REQUIREMENTS
	 * 
	 * Unpacks a file and verifies it as a Joomla element package
	 * Supports .gz .tar .tar.gz and .zip
	 *
	 * @param   string   $p_filename         The uploaded package filename or install directory
	 * @param   boolean  $alwaysReturnArray  If should return false (and leave garbage behind) or return $retval['type']=false
	 *
	 * @return  array|boolean  Array on success or boolean false on failure
	 *
	 * @since   3.1
	 */
	private function unpack($p_filename, $alwaysReturnArray = false)
	{
		// tmp directory
		$tmpdir_config = \JFactory::getConfig()->get('tmp_path');
		
		// Path to the archive
		$archivename = $tmpdir_config . '/' . $p_filename;
		// echo $archivename . "<br>";exit;

		// Temporary folder to extract the archive into
		$tmpdir = uniqid('install_');

		// Clean the paths to use for archive extraction
		$extractdir  = $tmpdir_config . '/'. $tmpdir;
		$archivename = \JPath::clean($archivename);
		
		// Do the unpacking of the archive
		try
		{
			$archive = new Archive(array('tmp_path' => $tmpdir_config));
			$extract = $archive->extract($archivename, $extractdir);
		}
		catch (\Exception $e)
		{
			if ($alwaysReturnArray)
			{
				return array(
					'extractdir'  => null,
					'packagefile' => $archivename,
					'type'        => false,
				);
			}

			return false;
		}

		if (!$extract)
		{
			if ($alwaysReturnArray)
			{
				return array(
					'extractdir'  => null,
					'packagefile' => $archivename,
					'type'        => false,
				);
			}

			return false;
		}
		
		if(JFile::exists($archivename)) {
			JFile::delete($archivename);
		}

		/*
		 * Let's set the extraction directory and package file in the result array so we can
		 * cleanup everything properly later on.
		 */
		$retval['extractdir'] = $extractdir;
		$retval['packagefile'] = $archivename;

		/*
		 * We have found the install directory so lets set it and then move on
		 * to detecting the extension type.
		 */
		$retval['dir'] = $extractdir;

		if ($alwaysReturnArray || $retval['type'])
		{
			return $retval;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Funktion kopiert nach dem Herunterladen und Entpacken die Dateien an ihren Bestimmungsort.
	 * Der tmp Ordner mit entpacktem Inhalt wird danach gelöscht
	 */ 
	private function copyContent($extractdir) {
		// Check extractdir exists
		if(!JFolder::exists($extractdir)) {
			throw new Exception("Extractdir not found");
		}
		
		// Check pro folder exists and delete to remove old files
		if(JFolder::exists($this->pro_folder_path)) {
			JFolder::delete($this->pro_folder_path);
		}
		
		$extractdir_pro = $extractdir . "/pro";
		
		//Target directory wird von ::copy selbst erzeugt
		JFolder::copy($extractdir_pro, $this->pro_folder_path);
		JFolder::delete($extractdir);
		
		return true;
	}
}